;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;
; $Id$
;
; Parser for R7RS syntax with R6RS and Larceny extensions.
;
; The scanner's state machine and the recursive descent parser
; were generated by Will Clinger's LexGen and ParseGen, so the
; parser can be extended or customized by regenerating those
; parts.
;
; LexGen and ParseGen are available at
; https://github.com/larcenists/larceny/tree/master/tools
;
; FIXME: some things are not yet implemented:
;
;     #!...!#    (unimportant)
;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;
; FIXME:
;
; The R6RS and R7RS standards specify different mnemonics for
; character names and mnemonic escapes within strings:
;
; Character names:
;
;     R6RS                R7RS
;     ----                ----
;     #\nul               #\null              (different)
;     #\alarm             #\alarm
;     #\backspace         #\backspace
;     #\tab               #\tab
;     #\linefeed
;     #\newline           #\newline
;     #\vtab
;     #\page
;     #\return            #\return
;     #\esc
;     #\space             #\space
;     #\delete            #\delete
;
; Mnemonic escapes within strings:
;
;     R6RS                R7RS
;     ----                ----
;     \a                  \a
;     \b                  \b
;     \t                  \t
;     \n                  \n
;     \v
;     \f
;     \r                  \r
;                         \|
;     \"                  \"
;     \\                  \\
;
; R7RS (small) erratum 5 gives us the impression those same
; mnemonic escapes are allowed within symbols, and we are
; proceeding under that assumption even though the errata
; don't yet confirm that impression.
;
; FIXME:
;
; This file contains several R6RS notations that could cause
; problems for cross-compilation:
;
; #\nul, #\backspace, #\tab, #\linefeed, #\vtab, and #\page.
;
; Larceny v0.93 and MzScheme v370 recognizes those notations.
;
; MzScheme v370 does not recognize #\alarm, #\esc, or #\delete,
; so they have been removed from this file.
;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;
; Larceny -- Scheme reader.
;
; install-reader takes no arguments
;
; install-reader assigns the following variables:
;
;       read
;       readtable-ref
;       readtable-set!
;       datum->source-location
;       datum-source-locations-clear!
;
; FIXME:  The readtable-ref and readtable-set! operations don't
; do anything except warn that they don't do anything.
;
; FIXME:  The old interface for obtaining source code locations,
; consisting of datum->source-location and datum->source-locations-clear!,
; is no longer supported.  The new interface is
;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;
; Source code locations.
;
;     (get-datum-with-source-locations input-port keep-source-locations?)
;
; If keep-source-locations? is false, then this procedure behaves
; just like get-datum.
;
; If keep-source-locations? is true, then this procedure returns
; two values.  The first return value is the value that would be
; returned by get-datum.  The second return value is an assocation
; list that maps every subform to the half-open interval consisting
; of its starting and ending positions, where each position is a
; vector of the form #(i j k), where
;
;     i is the 0-origin character position
;     j is the 0-origin line number
;     k is the 0-origin column number

($$trace "reader")

(define (install-reader)

  (define (deprecated-error who)
    (assertion-violation who "operation no longer supported"))

  (define (deprecated-warning who)
    (display "WARNING: " (current-error-port))
    (display who (current-error-port))
    (display " is no longer supported" (current-error-port)))

  (set! read
        (lambda p
          (let ((p (if (pair? p)
                       (car p)
                       (current-input-port))))
            (get-datum p))))

  (set! readtable-ref
        (lambda (char)
          (deprecated-error 'readtable-ref)))
 
  (set! readtable-set!
        (lambda (char l)
          (deprecated-error 'readtable-set!)))

  (set! datum->source-location
    (lambda (obj)
      (deprecated-warning 'datum->source-location)
      0))

  (set! datum-source-locations-clear!
    (lambda ()
      (deprecated-warning 'datum-source-locations-clear!)
      #t))
 
  #t)

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;
; Source code locations.
;
; Represented as a vector of three elements #(i j k) where
;     i is the 0-origin character position
;     j is the 0-origin line number
;     k is the 0-origin column number
;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(define (make-source-location input-port)
  (let* ((i (port-position-nocache input-port))
         (j (port-lines-read input-port))
         (k (- i (port-line-start input-port))))
    (vector i j k)))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;
; Fixup objects are used to implement SRFI 38.
;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(define rtd:fixup-object
  (make-rtd 'fixup-object
            '#((immutable index)
               (mutable ready)
               (mutable value))))

(define make-raw-fixup-object (rtd-constructor rtd:fixup-object))

(define (make-fixup-object index)
  (make-raw-fixup-object index #f #f))

(define fixup-object? (rtd-predicate rtd:fixup-object))

(define fixup-ready? (rtd-accessor rtd:fixup-object 'ready))

(define fixup-index (rtd-accessor rtd:fixup-object 'index))

(define fixup-value (rtd-accessor rtd:fixup-object 'value))

(define (fixup-ready! fixup obj)
  (raw-fixup-value! fixup obj)
  (raw-fixup-ready! fixup #t))

(define raw-fixup-ready! (rtd-mutator rtd:fixup-object 'ready))
(define raw-fixup-value! (rtd-mutator rtd:fixup-object 'value))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;; Historical note:
;;;
;;; As the R7RS lexical syntax was added during development of v0.98,
;;; the get-datum code grew so large that Petit Larceny couldn't
;;; use the v0.97 read procedure to compile reader.sch.
;;; That problem was related to an issue of space efficiency in
;;; the v0.97 implementation of get-datum itself.  The Standard-C
;;; assembler translated the MacScheme machine code for get-datum
;;; into a single C function whose textual representation was a
;;; single string of about 2185727 characters.  With Larceny's
;;; flat4 representation for Unicode strings, that becomes more
;;; than 8 megabytes.  When the expand-accumulator procedure
;;; tried to double the size of the token accumulator, it tried
;;; to allocate a string object containing more than 16 megabytes,
;;; which violated a hard limit in Larceny v0.97, causing a
;;; Larceny Panic.
;;;
;;; The regional garbage collector needs some such hard limit
;;; on the maximum size of a single object, so removing that
;;; 16-megabyte limit wouldn't be the answer even if it were
;;; easy to do.
;;;
;;; To solve the problem, we pulled the state machine out of
;;; the get-datum procedure so it can be compiled separately.
;;; Fortunately, the state machine refers to only a few
;;; procedures that manipulate the get-datum procedure's
;;; state.  (It also assigns to a local variable of get-datum,
;;; but a hack works around that.)
;;;
;;; Separating the state machine from the rest of the code for
;;; get-datum is something we needed to do anyway to make the
;;; read procedure easier to customize.
;;;
;;; The state machine is now defined in a separate file
;;; (reader-dfa.sch).  Even a small change to the regular
;;; expressions that describe Scheme tokens is likely to change
;;; thousands of lines in the generated code for the state
;;; machine, which makes the reader code look unstable when
;;; its version history is examined.
;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

;;; For the state machine that used to be here, see reader-dfa.sch

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;
; This is the real parser.
;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(define (get-datum input-port)
  (get-datum-with-source-locations input-port #f))

(define (get-datum-with-source-locations input-port keep-source-locations?)

  ; Constants and local variables.

  (let* (; Constants.

         ; initial length of string_accumulator

         (initial_accumulator_length 64)

         ; Encodings of error messages.

         (errLongToken 1)                 ; extremely long token
         (errIncompleteToken 2)      ; any lexical error, really
         (errIllegalHexEscape 3)                 ; illegal \x...
         (errIllegalNamedChar 4)                 ; illegal #\...
         (errIllegalString 5)           ; illegal string or text
         (errIllegalSymbol 6)                   ; illegal symbol
         (errIllegalBoolean 7)     ; disallowed #true or #!false
         (errIllegalSharing 8)     ; disallowed shared structure
         (errIllegalBytevector 9)          ; disallowed #u8(...)
         (errNoDelimiter 10)     ; missing delimiter after token
         (errSquareBracket 11)    ; square bracket when disabled
         (errBug 12)           ; bug in reader, shouldn't happen
         (errLexGenBug 13)                        ; can't happen
         (errIllegalText 14)         ; disallowed immutable text

         ; Named characters that MzScheme doesn't yet recognize.

         (char:alarm  (integer->char 7))
         (char:esc    (integer->char 27))
         (char:delete (integer->char 127))

         ; Important but unnamed non-Ascii characters.

         (char:nel    (integer->char #x85))
         (char:ls     (integer->char #x2028))

         ; State for one-token buffering in lexical analyzer.

         (kindOfNextToken 'z1)      ; valid iff nextTokenIsReady
         (nextTokenIsReady #f)

         (tokenValue "")  ; string associated with current token

         ; A string buffer for the characters of the current token.
         ; Resized as necessary.

         (string_accumulator (make-string initial_accumulator_length))

         ; Number of characters in string_accumulator.

         (string_accumulator_length 0)

         ; Source location for the start of the current token.

         (locationStart
          (if keep-source-locations?
              (make-source-location input-port)
              '#(0 0 0)))

         ; Stack of source locations.

         (locationStack '())

         ; Association list of data and their source locations.

         (locations '())

         ; This variable will be false until a shared-structure
         ; label (as in R7RS or SRFI 38) is encountered.
         ; Then it becomes a hash table.

         (shared-structures #f)

        )

    ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
    ;
    ; ParseGen generated the code for the strong LL(1) parser.
    ;
    ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
  
(define (parse-outermost-datum)
  (case (next-token)
    ((xsharpdot
       xfaslp
       xfaslg
       xbox
       unsyntaxsplicing
       unsyntax
       quasisyntax
       syntax
       splicing
       comma
       backquote
       quote
       lbracket
       lparen
       vecstart
       xfaslb
       bvecstart
       miscflag
       id
       xstring
       string
       text
       character
       xfaslf
       xfaslc
       number
       boolean
       sharingdef
       sharinguse)
     (let ((ast1 (parse-datum))) (identity ast1)))
    ((eofobj) (begin (consume-token!) (makeEOF)))
    (else
     (parse-error
       '<outermost-datum>
       '(backquote
          boolean
          bvecstart
          character
          comma
          eofobj
          id
          lbracket
          lparen
          miscflag
          number
          quasisyntax
          quote
          sharingdef
          sharinguse
          splicing
          string
          syntax
          text
          unsyntax
          unsyntaxsplicing
          vecstart
          xbox
          xfaslb
          xfaslc
          xfaslf
          xfaslg
          xfaslp
          xsharpdot
          xstring)))))

(define (parse-datum)
  (case (next-token)
    ((sharinguse)
     (let ((ast1 (parse-sharinguse)))
       (makeSharingUse ast1)))
    ((sharingdef)
     (let ((ast1 (parse-sharingdef)))
       (let ((ast2 (parse-datum)))
         (makeSharingDef ast1 ast2))))
    ((boolean
       number
       xfaslc
       xfaslf
       character
       text
       string
       xstring
       id
       miscflag
       bvecstart
       xfaslb
       vecstart
       lparen
       lbracket
       quote
       backquote
       comma
       splicing
       syntax
       quasisyntax
       unsyntax
       unsyntaxsplicing
       xbox
       xfaslg
       xfaslp
       xsharpdot)
     (let ((ast1 (parse-udatum))) (identity ast1)))
    (else
     (parse-error
       '<datum>
       '(backquote
          boolean
          bvecstart
          character
          comma
          id
          lbracket
          lparen
          miscflag
          number
          quasisyntax
          quote
          sharingdef
          sharinguse
          splicing
          string
          syntax
          text
          unsyntax
          unsyntaxsplicing
          vecstart
          xbox
          xfaslb
          xfaslc
          xfaslf
          xfaslg
          xfaslp
          xsharpdot
          xstring)))))

(define (parse-udatum)
  (case (next-token)
    ((xsharpdot)
     (begin
       (consume-token!)
       (let ((ast1 (parse-datum))) (sharpDot ast1))))
    ((xfaslp)
     (begin
       (consume-token!)
       (let ((ast1 (parse-datum))) (list2proc ast1))))
    ((xfaslg)
     (begin
       (consume-token!)
       (let ((ast1 (parse-symbol))) (sym2global ast1))))
    ((xbox unsyntaxsplicing
           unsyntax
           quasisyntax
           syntax
           splicing
           comma
           backquote
           quote
           lbracket
           lparen
           vecstart
           xfaslb
           bvecstart)
     (let ((ast1 (parse-location)))
       (let ((ast2 (parse-structured)))
         (makeStructured ast1 ast2))))
    ((miscflag) (begin (consume-token!) (makeFlag)))
    ((id) (begin (consume-token!) (makeSym)))
    ((xstring)
     (begin (consume-token!) (makeXString)))
    ((string) (begin (consume-token!) (makeString)))
    ((text) (begin (consume-token!) (makeText)))
    ((character) (begin (consume-token!) (makeChar)))
    ((xfaslf) (begin (consume-token!) (makeFlonum)))
    ((xfaslc) (begin (consume-token!) (makeCompnum)))
    ((number) (begin (consume-token!) (makeNum)))
    ((boolean) (begin (consume-token!) (makeBool)))
    (else
     (parse-error
       '<udatum>
       '(backquote
          boolean
          bvecstart
          character
          comma
          id
          lbracket
          lparen
          miscflag
          number
          quasisyntax
          quote
          splicing
          string
          syntax
          text
          unsyntax
          unsyntaxsplicing
          vecstart
          xbox
          xfaslb
          xfaslc
          xfaslf
          xfaslg
          xfaslp
          xsharpdot
          xstring)))))

(define (parse-structured)
  (case (next-token)
    ((bvecstart xfaslb)
     (let ((ast1 (parse-bytevector))) (identity ast1)))
    ((vecstart)
     (let ((ast1 (parse-vector))) (identity ast1)))
    ((lparen
       lbracket
       quote
       backquote
       comma
       splicing
       syntax
       quasisyntax
       unsyntax
       unsyntaxsplicing
       xbox)
     (let ((ast1 (parse-list))) (identity ast1)))
    (else
     (parse-error
       '<structured>
       '(backquote
          bvecstart
          comma
          lbracket
          lparen
          quasisyntax
          quote
          splicing
          syntax
          unsyntax
          unsyntaxsplicing
          vecstart
          xbox
          xfaslb)))))

(define (parse-symbol)
  (case (next-token)
    ((id) (begin (consume-token!) (makeSym)))
    (else (parse-error '<symbol> '(id)))))

(define (parse-list)
  (case (next-token)
    ((xbox unsyntaxsplicing
           unsyntax
           quasisyntax
           syntax
           splicing
           comma
           backquote
           quote)
     (let ((ast1 (parse-abbreviation)))
       (identity ast1)))
    ((lbracket)
     (begin
       (consume-token!)
       (let ((ast1 (parse-blst2))) (identity ast1))))
    ((lparen)
     (begin
       (consume-token!)
       (let ((ast1 (parse-list2))) (identity ast1))))
    (else
     (parse-error
       '<list>
       '(backquote
          comma
          lbracket
          lparen
          quasisyntax
          quote
          splicing
          syntax
          unsyntax
          unsyntaxsplicing
          xbox)))))

(define (parse-list2)
  (case (next-token)
    ((xsharpdot
       xfaslp
       xfaslg
       xbox
       unsyntaxsplicing
       unsyntax
       quasisyntax
       syntax
       splicing
       comma
       backquote
       quote
       lbracket
       lparen
       vecstart
       xfaslb
       bvecstart
       miscflag
       id
       xstring
       string
       text
       character
       xfaslf
       xfaslc
       number
       boolean
       sharingdef
       sharinguse)
     (let ((ast1 (parse-datum)))
       (let ((ast2 (parse-list3))) (cons ast1 ast2))))
    ((rparen) (begin (consume-token!) (emptyList)))
    (else
     (parse-error
       '<list2>
       '(backquote
          boolean
          bvecstart
          character
          comma
          id
          lbracket
          lparen
          miscflag
          number
          quasisyntax
          quote
          rparen
          sharingdef
          sharinguse
          splicing
          string
          syntax
          text
          unsyntax
          unsyntaxsplicing
          vecstart
          xbox
          xfaslb
          xfaslc
          xfaslf
          xfaslg
          xfaslp
          xsharpdot
          xstring)))))

(define (parse-list3)
  (case (next-token)
    ((rparen
       period
       sharinguse
       sharingdef
       boolean
       number
       xfaslc
       xfaslf
       character
       text
       string
       xstring
       id
       miscflag
       bvecstart
       xfaslb
       vecstart
       lparen
       lbracket
       quote
       backquote
       comma
       splicing
       syntax
       quasisyntax
       unsyntax
       unsyntaxsplicing
       xbox
       xfaslg
       xfaslp
       xsharpdot)
     (let ((ast1 (parse-data)))
       (let ((ast2 (parse-list4)))
         (pseudoAppend ast1 ast2))))
    (else
     (parse-error
       '<list3>
       '(backquote
          boolean
          bvecstart
          character
          comma
          id
          lbracket
          lparen
          miscflag
          number
          period
          quasisyntax
          quote
          rparen
          sharingdef
          sharinguse
          splicing
          string
          syntax
          text
          unsyntax
          unsyntaxsplicing
          vecstart
          xbox
          xfaslb
          xfaslc
          xfaslf
          xfaslg
          xfaslp
          xsharpdot
          xstring)))))

(define (parse-list4)
  (case (next-token)
    ((period)
     (begin
       (consume-token!)
       (let ((ast1 (parse-datum)))
         (if (eq? (next-token) 'rparen)
             (begin (consume-token!) (identity ast1))
             (parse-error '<list4> '(rparen))))))
    ((rparen) (begin (consume-token!) (emptyList)))
    (else (parse-error '<list4> '(period rparen)))))

(define (parse-blst2)
  (case (next-token)
    ((xsharpdot
       xfaslp
       xfaslg
       xbox
       unsyntaxsplicing
       unsyntax
       quasisyntax
       syntax
       splicing
       comma
       backquote
       quote
       lbracket
       lparen
       vecstart
       xfaslb
       bvecstart
       miscflag
       id
       xstring
       string
       text
       character
       xfaslf
       xfaslc
       number
       boolean
       sharingdef
       sharinguse)
     (let ((ast1 (parse-datum)))
       (let ((ast2 (parse-blst3))) (cons ast1 ast2))))
    ((rbracket) (begin (consume-token!) (emptyList)))
    (else
     (parse-error
       '<blst2>
       '(backquote
          boolean
          bvecstart
          character
          comma
          id
          lbracket
          lparen
          miscflag
          number
          quasisyntax
          quote
          rbracket
          sharingdef
          sharinguse
          splicing
          string
          syntax
          text
          unsyntax
          unsyntaxsplicing
          vecstart
          xbox
          xfaslb
          xfaslc
          xfaslf
          xfaslg
          xfaslp
          xsharpdot
          xstring)))))

(define (parse-blst3)
  (case (next-token)
    ((rbracket
       period
       sharinguse
       sharingdef
       boolean
       number
       xfaslc
       xfaslf
       character
       text
       string
       xstring
       id
       miscflag
       bvecstart
       xfaslb
       vecstart
       lparen
       lbracket
       quote
       backquote
       comma
       splicing
       syntax
       quasisyntax
       unsyntax
       unsyntaxsplicing
       xbox
       xfaslg
       xfaslp
       xsharpdot)
     (let ((ast1 (parse-data)))
       (let ((ast2 (parse-blst4)))
         (pseudoAppend ast1 ast2))))
    (else
     (parse-error
       '<blst3>
       '(backquote
          boolean
          bvecstart
          character
          comma
          id
          lbracket
          lparen
          miscflag
          number
          period
          quasisyntax
          quote
          rbracket
          sharingdef
          sharinguse
          splicing
          string
          syntax
          text
          unsyntax
          unsyntaxsplicing
          vecstart
          xbox
          xfaslb
          xfaslc
          xfaslf
          xfaslg
          xfaslp
          xsharpdot
          xstring)))))

(define (parse-blst4)
  (case (next-token)
    ((period)
     (begin
       (consume-token!)
       (let ((ast1 (parse-datum)))
         (if (eq? (next-token) 'rbracket)
             (begin (consume-token!) (identity ast1))
             (parse-error '<blst4> '(rbracket))))))
    ((rbracket) (begin (consume-token!) (emptyList)))
    (else (parse-error '<blst4> '(period rbracket)))))

(define (parse-abbreviation)
  (case (next-token)
    ((quote backquote
            comma
            splicing
            syntax
            quasisyntax
            unsyntax
            unsyntaxsplicing
            xbox)
     (let ((ast1 (parse-abbrev-prefix)))
       (let ((ast2 (parse-datum))) (list ast1 ast2))))
    (else
     (parse-error
       '<abbreviation>
       '(backquote
          comma
          quasisyntax
          quote
          splicing
          syntax
          unsyntax
          unsyntaxsplicing
          xbox)))))

(define (parse-abbrev-prefix)
  (case (next-token)
    ((xbox) (begin (consume-token!) (symBox)))
    ((unsyntaxsplicing)
     (begin (consume-token!) (symUnsyntax-splicing)))
    ((unsyntax)
     (begin (consume-token!) (symUnsyntax)))
    ((quasisyntax)
     (begin (consume-token!) (symQuasisyntax)))
    ((syntax) (begin (consume-token!) (symSyntax)))
    ((splicing)
     (begin (consume-token!) (symSplicing)))
    ((comma) (begin (consume-token!) (symUnquote)))
    ((backquote)
     (begin (consume-token!) (symBackquote)))
    ((quote) (begin (consume-token!) (symQuote)))
    (else
     (parse-error
       '<abbrev-prefix>
       '(backquote
          comma
          quasisyntax
          quote
          splicing
          syntax
          unsyntax
          unsyntaxsplicing
          xbox)))))

(define (parse-vector)
  (case (next-token)
    ((vecstart)
     (begin
       (consume-token!)
       (let ((ast1 (parse-data)))
         (if (eq? (next-token) 'rparen)
             (begin (consume-token!) (list2vector ast1))
             (parse-error '<vector> '(rparen))))))
    (else (parse-error '<vector> '(vecstart)))))

(define (parse-bytevector)
  (case (next-token)
    ((xfaslb)
     (begin (consume-token!) (makeCodevector)))
    ((bvecstart)
     (begin
       (consume-token!)
       (let ((ast1 (parse-octets)))
         (if (eq? (next-token) 'rparen)
             (begin (consume-token!) (list2bytevector ast1))
             (parse-error '<bytevector> '(rparen))))))
    (else
     (parse-error '<bytevector> '(bvecstart xfaslb)))))

(define (parse-data)
  (case (next-token)
    ((xsharpdot
       xfaslp
       xfaslg
       xbox
       unsyntaxsplicing
       unsyntax
       quasisyntax
       syntax
       splicing
       comma
       backquote
       quote
       lbracket
       lparen
       vecstart
       xfaslb
       bvecstart
       miscflag
       id
       xstring
       string
       text
       character
       xfaslf
       xfaslc
       number
       boolean
       sharingdef
       sharinguse)
     (let ((ast1 (parse-datum)))
       (let ((ast2 (parse-data))) (cons ast1 ast2))))
    ((rparen period rbracket) (emptyList))
    (else
     (parse-error
       '<data>
       '(backquote
          boolean
          bvecstart
          character
          comma
          id
          lbracket
          lparen
          miscflag
          number
          period
          quasisyntax
          quote
          rbracket
          rparen
          sharingdef
          sharinguse
          splicing
          string
          syntax
          text
          unsyntax
          unsyntaxsplicing
          vecstart
          xbox
          xfaslb
          xfaslc
          xfaslf
          xfaslg
          xfaslp
          xsharpdot
          xstring)))))

(define (parse-octets)
  (case (next-token)
    ((number)
     (let ((ast1 (parse-octet)))
       (let ((ast2 (parse-octets))) (cons ast1 ast2))))
    ((rparen) (emptyList))
    (else (parse-error '<octets> '(number rparen)))))

(define (parse-octet)
  (case (next-token)
    ((number) (begin (consume-token!) (makeOctet)))
    (else (parse-error '<octet> '(number)))))

(define (parse-location)
  (case (next-token)
    ((xbox unsyntaxsplicing
           unsyntax
           quasisyntax
           syntax
           splicing
           comma
           backquote
           quote
           lbracket
           lparen
           vecstart
           xfaslb
           bvecstart)
     (sourceLocation))
    (else
     (parse-error
       '<location>
       '(backquote
          bvecstart
          comma
          lbracket
          lparen
          quasisyntax
          quote
          splicing
          syntax
          unsyntax
          unsyntaxsplicing
          vecstart
          xbox
          xfaslb)))))

(define (parse-sharingdef)
  (case (next-token)
    ((sharingdef)
     (begin (consume-token!) (sharingDef)))
    (else (parse-error '<sharingdef> '(sharingdef)))))

(define (parse-sharinguse)
  (case (next-token)
    ((sharinguse)
     (begin (consume-token!) (sharingUse)))
    (else (parse-error '<sharinguse> '(sharinguse)))))

    ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
    ;
    ; End of LL(1) parser generated by ParseGen.
    ;
    ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

    ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
    ;
    ; Lexical analyzer.
    ;
    ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
  
    ; next-token and consume-token! are called by the parser.
  
    ; Returns the current token.
  
    (define (next-token)
      (if nextTokenIsReady
          kindOfNextToken
          (begin (set! string_accumulator_length 0)
                 (scanner0))))
  
    ; Consumes the current token.
  
    (define (consume-token!)
      (set! nextTokenIsReady #f))
  
    ; Called by the lexical analyzer's state machine.
  
    (define (scannerError msg)
      (define msgtxt
        (cond ((= msg errLongToken)
               "amazingly long token")
              ((= msg errIncompleteToken)
               "incomplete or illegal token")
              ((= msg errIllegalHexEscape)
               "illegal hex escape")
              ((= msg errIllegalNamedChar)
               "illegal character syntax")
              ((= msg errIllegalString)
               "illegal string or text syntax")
              ((= msg errIllegalText)
               "illegal immutable text notation (strict R6RS mode)")
              ((= msg errIllegalSymbol)
               "illegal symbol syntax")
              ((= msg errIllegalBoolean)
               "illegal boolean (strict R6RS mode)")
              ((= msg errIllegalSharing)
               "illegal SRFI 38 notation (strict R6RS mode)")
              ((= msg errIllegalBytevector)
               "illegal bytevector (strict R6RS mode)")
              ((= msg errNoDelimiter)
               "missing delimiter")
              ((= msg errSquareBracket)
               "square brackets are disabled (as in strict R7RS mode)")
              ((= msg errLexGenBug)
               "bug in lexical analyzer (generated)")
              (else "bug in lexical analyzer")))
      (let* ((c (scanChar))
             (next (if (char? c) (string c) ""))
             (line (+ 1 (port-lines-read input-port)))
             (msgtxt (string-append msgtxt
                                    " in line "
                                    (number->string line)))
             (irritant1 (substring string_accumulator
                                   0
                                   string_accumulator_length))
             (msgtxt (string-append msgtxt
                                    ": "
                                    irritant1
                                    next)))

        ; must avoid infinite loop on current input port
        ;
        ; FIXME: the R6RS says the exception must (in some cases)
        ; be both &lexical and &i/o-read, but the &i/o-read
        ; part appears to have been a mistake.

        (consumeChar)
        (raise-r6rs-exception
         (make-lexical-violation)
         'get-datum
         (string-append "lexical error: " msgtxt " ")
         (list irritant1 next input-port)))
      (next-token))
  
    ; Accepts a token of the given kind, returning that kind.
    ;
    ; For some kinds of tokens, a value for the token must also
    ; be recorded in tokenValue.  Most of those tokens must be
    ; followed by a delimiter.
    ;
    ; Some magical tokens require special processing.
  
    (define (accept t)
      (case t

       ((comment)
        ; The token is #|, which starts a nested comment.
        (scan-nested-comment)
        (next-token))

       ((commentdatum)
        ; The token is #; so parse and ignore the next datum.
        (parse-datum)
        (next-token))

       ;; R6RS forbids #u8(...)

       ((bvecstart)
        (if (and (char=? #\u (string-ref string_accumulator 1))
                 (not (r7rs-weirdness?))
                 (not (larceny-weirdness?))
                 (not (traditional-weirdness?)))
            (scannerError errIllegalBytevector)
            (begin (set! kindOfNextToken t)
                   (set! nextTokenIsReady #t)
                   t)))

       ;; R6RS forbids SRFI 38 and R7RS notation for shared structure

       ((sharingdef sharinguse)
        (set! tokenValue
              (substring string_accumulator
                         0 string_accumulator_length))
        (if (and (not (r7rs-weirdness?))
                 (not (larceny-weirdness?))
                 (not (traditional-weirdness?)))
            (scannerError errIllegalSharing)
            (begin (set! kindOfNextToken t)
                   (set! nextTokenIsReady #t)
                   t)))

       ((id boolean number character string text miscflag period)

        (set! tokenValue
              (substring string_accumulator
                         0 string_accumulator_length))

        (cond ((and (eq? t 'miscflag)
                    (member tokenValue
                            '("#!r7rs"
                              "#!r6rs"
                              "#!r5rs"
                              "#!larceny"
                              "#!err5rs"
                              "#!fold-case"
                              "#!no-fold-case")))
               (set-mode! (string->symbol
                           (substring tokenValue
                                      2
                                      (string-length tokenValue))))
               (next-token))

              ((or (delimiter? (scanChar))
                   (eq? t 'string)
                   (eq? t 'text))
               (set! kindOfNextToken t)
               (set! nextTokenIsReady #t)
               t)

              (else
               (scannerError errNoDelimiter))))

       ; Yes, we really need to disable square brackets because
       ; they might come to mean something else in R7RS large.

       ((lbracket)
        (if (or (r6rs-weirdness?)
                (read-square-bracket-as-paren))
            (begin (set! kindOfNextToken t)
                   (set! nextTokenIsReady #t)
                   t)
            (scannerError errSquareBracket)))

       (else
        (set! kindOfNextToken t)
        (set! nextTokenIsReady #t)
        t)))

    ; Having seen a #| token, scans and discards the entire comment.

    (define (scan-nested-comment)
      (define (loop depth)
        (let ((c (scanChar)))
          (cond ((= depth 0) #t)
                ((eof-object? c)
                 (scannerError errIncompleteToken))
                ((char=? c #\#)
                 (consumeChar)
                 (if (char=? (scanChar) #\|)
                     (begin (consumeChar) (loop (+ depth 1)))
                     (loop depth)))
                ((char=? c #\|)
                 (consumeChar)
                 (if (char=? (scanChar) #\#)
                     (begin (consumeChar) (loop (- depth 1)))
                     (loop depth)))
                (else
                 (consumeChar)
                 (loop depth)))))
      (loop 1))

    ; Most reader modes are now port-specific.
    ; Some, but not all, can be changed by reading a flag.

    (define (set-mode! m)
      (let ()
        (case m
         ((r6rs)
          (set-port-switches input-port
                             port-folds-case! #f
                             io/port-allows-r6rs-weirdness!        #t
                             io/port-allows-r7rs-weirdness!        #f
                             io/port-allows-larceny-weirdness!     #f
                             io/port-allows-traditional-weirdness! #f
                             io/port-allows-mzscheme-weirdness!    #f
                             io/port-recognizes-javadot-symbols!   #f))

         ((r7rs)
          (set-port-switches input-port
                             port-folds-case! #f
                             io/port-allows-r7rs-weirdness!    #t))

         ((err5rs)
          (set-port-switches input-port
                             io/port-allows-r6rs-weirdness!    #t
                             io/port-allows-r7rs-weirdness!    #t
                             io/port-allows-larceny-weirdness! #t))

         ((r5rs)
          (set-port-switches input-port
                             port-folds-case! #t
                             io/port-allows-r7rs-weirdness!    #t
                             io/port-allows-larceny-weirdness! #t))

         ((fasl larceny)
          (set-port-switches input-port
                             port-folds-case! #f
                             io/port-allows-r6rs-weirdness!    #t
                             io/port-allows-r7rs-weirdness!    #t
                             io/port-allows-larceny-weirdness! #t))

         ((fold-case)
          (port-folds-case! input-port #t))

         ((no-fold-case)
          (port-folds-case! input-port #f))

         ; FIXME: these compiler switches might not be defined in all heaps.

         ;((slow)
         ; (set-switches compiler-switches 'slow))
         ;((fast)
         ; (set-switches compiler-switches 'fast-safe))
         ;((safe)
         ; (set-switches runtime-safety-checking #t
         ;               catch-undefined-globals #t))
         ;((unsafe)
         ; (set-switches compiler-switches 'fast-unsafe))

         (else #t))))

    ; This looks weird but saves code space.

    (define (set-port-switches port . settings)
      (do ((settings settings (cddr settings)))
          ((null? settings))
        ((car settings) port (cadr settings))))

    ; This makes source code lines shorter.

    (define (r5rs-weirdness?)
      (io/port-allows-r5rs-weirdness? input-port))
  
    (define (r6rs-weirdness?)
      (io/port-allows-r6rs-weirdness? input-port))
  
    (define (r7rs-weirdness?)
      (io/port-allows-r7rs-weirdness? input-port))
  
    (define (larceny-weirdness?)
      (io/port-allows-larceny-weirdness? input-port))
  
    (define (traditional-weirdness?)
      (io/port-allows-traditional-weirdness? input-port))
  
    (define (mzscheme-weirdness?)
      (io/port-allows-mzscheme-weirdness? input-port))
  
    ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
    ;
    ; Character i/o, so to speak.
    ; Uses the input-port as input.
    ;
    ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
  
    (define (scanChar)
      (peek-char input-port))

    ; Consumes the current character.  Returns unspecified values.
  
    (define (consumeChar)
      (if (< string_accumulator_length (string-length string_accumulator))
          (let ((c (read-char input-port)))
            (if (char? c)
                (begin
                 (string-set! string_accumulator
                              string_accumulator_length
                              c)
                 (set! string_accumulator_length
                       (+ string_accumulator_length 1)))))
          (begin (expand-accumulator) (consumeChar))))

    ; Doubles the size of string_accumulator while
    ; preserving its contents.
    ;
    ; Insanely long tokens are not supported.

    (define (expand-accumulator)
      (define maximum 4000000)
      (let* ((n (string-length string_accumulator))
             (newn (* 2 n))
             (newn (cond ((< newn maximum)
                          newn)
                         ((< n maximum)
                          maximum)
                         (else
                          (scannerError errLongToken))))
             (new (make-string newn)))
        (do ((i 0 (+ i 1)))
            ((= i n))
          (string-set! new i (string-ref string_accumulator i)))
        (set! string_accumulator new)))

    (define (record-source-location x start)
      (if keep-source-locations?
          (set! locations
                (cons (vector x start (make-source-location input-port))
                      locations)))
      x)
  
    ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
    ;
    ; Miscellaneous utility routines.
    ;
    ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

    ; Determines whether its argument is a <delimiter>.

    (define (delimiter? c)
      (case c
       ((#\( #\) #\[ #\] #\" #\; #\#)
        #t)
       ((#\#)
        (r6rs-weirdness?))
       (else
        (or (not (char? c))
            (char-whitespace? c)
            (= (char->integer c) #x00ab)
            (= (char->integer c) #x00bb)))))

    ; Given the integer parsed from a hex escape,
    ; returns the corresponding Unicode character.

    (define (checked-integer->char n)
      (if (or (< n #xd800)
              (<= #xe000 n #x10ffff))
          (integer->char n)
          (scannerError errIllegalHexEscape)))

    ; Given a string and the index at the beginning of nonempty
    ; sequence of hexadecimal characters followed by a semicolon,
    ; returns two values:
    ;     the numerical value of the hex characters
    ;     the index following the semicolon

    (define (hex-escape s i)
      (let ((n (string-length s)))
        (define (loop i val)
          (if (>= i n)
              (scannerError errIllegalHexEscape)
              (let ((c (string-ref s i)))
                (case c
                 ((#\0 #\1 #\2 #\3 #\4 #\5 #\6 #\7 #\8 #\9)
                  (loop (+ i 1)
                        (+ (* 16 val)
                           (- (char->integer c) (char->integer #\0)))))
                 ((#\a #\b #\c #\d #\e #\f)
                  (loop (+ i 1)
                        (+ (* 16 val)
                           10
                           (- (char->integer c) (char->integer #\a)))))
                 ((#\A #\B #\C #\D #\E #\F)
                  (loop (+ i 1)
                        (+ (* 16 val)
                           10
                           (- (char->integer c) (char->integer #\A)))))
                 ((#\;)
                  (values val (+ i 1)))
                 (else (scannerError errIllegalHexEscape))))))
        (loop i 0)))
  
    ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
    ;
    ; Action procedures called by the parser.
    ;
    ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

    (define (emptyList) '())
  
    (define (identity x) x)

    (define (list2bytevector octets) (u8-list->bytevector octets))

    (define (list2proc vals)
      (if (larceny-weirdness?)
          (list->procedure vals)
          (parse-error '<datum> datum-starters)))

    (define (list2vector vals) (list->vector vals))
  
    (define (makeBool)
      (let ((x (case (string-ref tokenValue 1)
                ((#\t #\T) #t)
                ((#\f #\F) #f)
                (else (scannerError errBug)))))
        (if (and (< 2 (string-length tokenValue))
                 (not (r7rs-weirdness?)))
            (scannerError errIllegalBoolean)
            (record-source-location x locationStart))))
  
    (define (makeChar)
      (let* ((n (string-length tokenValue))
             (x (cond ((= n 3)
                       (string-ref tokenValue 2))
                      ((char=? #\x (string-ref tokenValue 2))
                       (checked-integer->char
                        (string->number (substring tokenValue 3 n) 16)))
                      (else
                       (let* ((s (substring tokenValue 2 n))
                              (s (if (port-folds-case? input-port)
                                     (string-foldcase s)
                                     s))
                              (sym (string->symbol s)))
                         (define (illegal)
                           (scannerError errIllegalNamedChar))
                         (case sym
                          ((alarm)             char:alarm)
                          ((backspace)         #\backspace)
                          ((tab)               #\tab)
                          ((newline)           #\linefeed)
                          ((return)            #\return)
                          ((space)             #\space)
                          ((delete)            char:delete)
                          ((nul)
                           (if (r6rs-weirdness?)
                               #\nul
                               (illegal)))
                          ((null)
                           (if (r5rs-weirdness?)
                               #\nul
                               (illegal)))
                          ((linefeed)
                           (if (r6rs-weirdness?)
                               #\linefeed
                               (illegal)))
                          ((vtab)
                           (if (r6rs-weirdness?)
                               #\vtab
                               (illegal)))
                          ((page)
                           (if (r6rs-weirdness?)
                               #\page
                               (illegal)))
                          ((esc)
                           (if (r6rs-weirdness?)
                               #\esc
                               (illegal)))
                          ((escape)
                           (if (r7rs-weirdness?)
                               #\esc
                               (illegal)))
                          (else
                           (scannerError errIllegalNamedChar))))))))
        (record-source-location x locationStart)))

    ; #^B"..."
    ; Coding bytevectors as strings is inherently evil.
    ; Must keep this in sync with print-slashed-bytevector,
    ; which is defined in Lib/Common/print.sch.
    ; All characters represent bytes except for #\" and #\\.
    ; The #\" begins and terminates the sequence of bytes,
    ; and the #\\ is used only to escape the next character.
    ; Note: This code doesn't check for end of file, and
    ; shouldn't have to.

    (define (makeCodevector)
      (define (loop)
        (let ((c (scanChar)))
          ; The call to consumeChar puts the char in string_accumulator.
          (consumeChar)
          (case c
           ((#\")
            (let* ((n (- string_accumulator_length 1)) ; ignore closing #\"
                   (bv (make-bytevector n)))
              (do ((i 0 (+ i 1)))
                  ((= i n) bv)
                (bytevector-set! bv i (char->integer
                                       (string-ref string_accumulator i))))))
           ((#\\)
            ; Erase the escape character.
            (set! string_accumulator_length (- string_accumulator_length 1))
            (consumeChar)
            (loop))
           (else
            (loop)))))
      (if (larceny-weirdness?)
          (let ((c (scanChar)))
            (if (char=? c #\")
                (begin (consumeChar)
                       (set! string_accumulator_length 0)
                       (let ((bv (loop)))
			 (sys$codevector-iflush bv)
			 bv))
                (parse-error '<datum> datum-starters)))
          (parse-error '<datum> datum-starters)))
  
    ; #^Cxxxxxxxxxxxxxxxx
    ; Coding bits as characters is inherently evil.
    ; It forces .fasl files to be encoded as Latin-1.

    (define (makeCompnum)
      (define (loop bv i n)
        (if (< i n)
            (let* ((c (scanChar))
                   (b (char->integer c)))
              (consumeChar)
              (bytevector-set! bv i b)
              (loop bv (+ i 1) n))))
      (if (larceny-weirdness?)
          (let ((z (make-bytevector 20)))
            (loop z 4 20)
            (typetag-set! z sys$tag.compnum-typetag)
            z)
          (parse-error '<datum> datum-starters)))

    (define (makeEOF) (eof-object))

    (define (makeFlag)

      ; The draft R6RS allows implementation-specific extensions
      ; of the form #!..., which are processed here.
      ; Note that the #!r6rs flag is a comment, handled by accept,
      ; so that flag will never be seen here.
      ; As of v0.98, the #!r7rs, #!r5rs, #!larceny,
      ; #!fold-case, and #!no-fold-case flags are also comments.

      (if (io/port-allows-flags? input-port)

          (let* ((n (string-length tokenValue))
                 (flag (string->symbol (substring tokenValue 2 n)))
                 (x (case flag
                     ((; fold-case no-fold-case        ; these are now comments
                       ; r7rs r6rs r5rs larceny        ; these are now comments
                       ; err5rs                        ; these are now comments
                       slow fast safe unsafe)
                      (set-mode! flag)
                      (unspecified))
                     ((fasl)
                      (set-mode! flag)
                      ((fasl-evaluator)))
                     ((unspecified) (unspecified))
                     ((undefined)   (undefined))
                     ((null)        '())
                     ((false)       #f)
                     ((true)        #t)
                     (else
                      (accept 'miscflag)
                      (parse-error '<miscflag> '(miscflag))))))
            (record-source-location x locationStart))

          (begin (accept 'miscflag)
                 (parse-error '<miscflag> '(miscflag)))))
  
    ; #^Fxxxxxxxx
    ; Coding bits as characters is inherently evil.
    ; It forces .fasl files to be encoded as Latin-1.

    (define (makeFlonum)
      (define (loop bv i n)
        (if (< i n)
            (let* ((c (scanChar))
                   (b (char->integer c)))
              (consumeChar)
              (bytevector-set! bv i b)
              (loop bv (+ i 1) n))))
      (if (larceny-weirdness?)
          (let ((x (make-bytevector 12)))
            (loop x 4 12)
            (typetag-set! x sys$tag.flonum-typetag)
            x)
          (parse-error '<datum> datum-starters)))

    (define (makeNum)
      (let ((x (string->number tokenValue)))
        (cond (x
               (record-source-location x locationStart))
              ((and (r7rs-weirdness?)
                    (string->number (string-downcase tokenValue)))
               =>
               (lambda (x)
                 (record-source-location x locationStart)))
              (else (accept 'number)
                    (parse-error '<number> '(number))))))
  
    (define (makeOctet)
      (let ((n (string->number tokenValue)))
        (if (and (exact? n) (integer? n) (<= 0 n 255))
            (record-source-location n locationStart)
            (begin (accept 'octet)
                   (parse-error '<octet> '(octet))))))
  
    (define (makeString)
      (makeStringOrText-common 'string))

    ;; Almost all of the processing for strings and texts is the same.
    ;; The only difference is what's done with the processed contents.
    ;; The argument is a symbol: string or text

    (define (makeStringOrText-common which)

      ; Must strip off outer double quotes and deal with escapes,
      ; which differ between R7RS and R6RS.
      ;
      ; i is the next index into tokenValue
      ; n is the exclusive upper bound for i
      ; newstring is a string that might become the result
      ; j is the next index into newstring

      (define (loop i n newstring j)
        (if (>= i n)
            (if (= j (string-length newstring))
                newstring
                (substring newstring 0 j))
            (let ((c (string-ref tokenValue i)))
              (cond ((or (char=? c #\return)
                         (char=? c #\linefeed)
                         (char=? c char:nel)
                         (char=? c char:ls))
                     (string-set! newstring j #\linefeed)
                     (let* ((i+1 (+ i 1))
                            (i+1 (if (and (char=? c #\return)
                                          (< i+1 n))
                                     (let ((c2 (string-ref tokenValue i+1)))
                                       (if (or (char=? c2 #\linefeed)
                                               (char=? c2 char:nel))
                                           (+ i 2)
                                           i+1))
                                     i+1)))
                       (loop i+1 n newstring (+ j 1))))
                    ((char=? c #\\)
                     (if (< (+ i 1) n)
                         (let ((c2 (string-ref tokenValue (+ i 1))))
                           (define (illegal)
                             (scannerError errIllegalString))
                           (case c2
                            ((#\a)
                             (string-set! newstring j char:alarm)
                             (loop (+ i 2) n newstring (+ j 1)))
                            ((#\b)
                             (string-set! newstring j #\backspace)
                             (loop (+ i 2) n newstring (+ j 1)))
                            ((#\t)
                             (string-set! newstring j #\tab)
                             (loop (+ i 2) n newstring (+ j 1)))
                            ((#\n)
                             (string-set! newstring j #\linefeed)
                             (loop (+ i 2) n newstring (+ j 1)))
                            ((#\v)
                             (if (r6rs-weirdness?)
                                 (begin (string-set! newstring j #\vtab)
                                        (loop (+ i 2) n newstring (+ j 1)))
                                 (illegal)))
                            ((#\" #\\)
                             (string-set! newstring j c2)
                             (loop (+ i 2) n newstring (+ j 1)))
                            ((#\f)
                             (if (r6rs-weirdness?)
                                 (begin (string-set! newstring j #\page)
                                        (loop (+ i 2) n newstring (+ j 1)))
                                 (illegal)))
                            ((#\r)
                             (string-set! newstring j #\return)
                             (loop (+ i 2) n newstring (+ j 1)))
                            ((#\|)
                             (if (r7rs-weirdness?)
                                 (begin (string-set! newstring j #\|)
                                        (loop (+ i 2) n newstring (+ j 1)))
                                 (illegal)))
                            ((#\x)
                             (call-with-values
                              (lambda () (hex-escape tokenValue (+ i 2)))
                              (lambda (sv i)
                                (string-set! newstring
                                             j
                                             (checked-integer->char sv))
                                (loop i n newstring (+ j 1)))))
                            (else
                             (ignore-escaped-line-ending (+ i 1)
                                                         n newstring j #f))))
                         (scannerError errIllegalString)))
                    (else
                     (string-set! newstring j c)
                     (loop (+ i 1) n newstring (+ j 1)))))))

      ; Ignores <intraline whitespace>* <line ending> <intraline whitespace>*
      ; after? is true iff the <line ending> has already been ignored.
      ; The other arguments are the same as for loop above.

      (define (ignore-escaped-line-ending i n newstring j after?)
        (cond ((< i n)
               (let ((c (string-ref tokenValue i)))
                 (cond ((or (char=? c #\tab)
                            (eq? 'Zs (char-general-category c)))
                        (ignore-escaped-line-ending (+ i 1)
                                                    n newstring j after?))
                       (after?
                        (loop i n newstring j))
                       ((or (char=? c #\return)
                            (char=? c #\linefeed)
                            (char=? c char:nel)
                            (char=? c char:ls))
                        (let* ((i+1 (+ i 1))
                               (i+1 (if (and (char=? c #\return)
                                             (< i+1 n))
                                        (let ((c2 (string-ref
                                                   tokenValue i+1)))
                                          (if (or (char=? c2 #\linefeed)
                                                  (char=? c2 char:nel))
                                              (+ i 2)
                                              i+1))
                                        i+1)))
                          (ignore-escaped-line-ending i+1 n newstring j #t)))
                       ((larceny-weirdness?)
                        (string-set! newstring j c)
                        (loop (+ i 1) n newstring (+ j 1)))
                       ((and (r7rs-weirdness?)
                             (let ((sv (char->integer c)))
                               ;; left and right double angle quotes
                               (or (= sv #x00ab)
                                   (= sv #x00bb))))
                        (string-set! newstring j c)
                        (loop (+ i 1) n newstring (+ j 1)))
                       (else
                        (scannerError errIllegalString)))))
              (after?
               (loop i n newstring j))
              (else
               (scannerError errIllegalString))))

      (let* ((n (string-length tokenValue))
             (s (loop 1 (- n 1) (make-string (- n 2)) 0))
             (x (case which
                 ((string) s)
                 ((text) (string->text s)))))
        (record-source-location x locationStart)))

    (define (makeStructured loc0 x)
      (record-source-location x loc0))

    ;; The R7RS syntax for symbols is equivalent to
    ;;
    ;; <identifier>  ::=  <identifier-prefix> <subsequent>*
    ;;                 |  <vline> <symbol element>* <vline>
    ;;                 |  <explicit sign>
    ;;
    ;; <identifier-prefix>  ::=  <initial>
    ;;                        |  <explicit sign> <sign subsequent>
    ;;                        |  <explicit sign> . <sign subsequent>
    ;;                        |  <explicit sign> . .
    ;;                        |  . <sign subsequent>
    ;;                        |  . .
    ;;
    ;; For R6RS, we add -> as another <identifier-prefix>.
    ;;
    ;; Larceny allows these extensions:
    ;;     leading . or @ or +: or -:
    ;;     vertical bars at beginning and end (as in R7RS)
    ;;     vertical bars (embedded, which do not quote)
    ;;     backslashes for other than hex or mnemonic escapes
    ;;         (note: they disable *all* case folding)
    ;;     MzScheme randomness
    ;;     several peculiar identifiers

    (define (makeSym)
      (let ((n (string-length tokenValue)))

        (define (return sym)
          (let ((x (if (and (io/port-recognizes-javadot-symbols? input-port)
                            (javadot-syntax? sym))
                       (symbol->javadot-symbol! sym)
                       sym)))
            (record-source-location x locationStart)))

        (define (identifier-prefix)
          (let ((c (string-ref tokenValue 0)))
            (cond 
                  ((char=? c #\\)
                   (if (or (larceny-weirdness?)
                           (r6rs-weirdness?))
                       (loop 0)
                       (scannerError errIllegalSymbol)))
                  ((char=? c #\@)
                   (if (or (larceny-weirdness?)
                           (r5rs-weirdness?)
                           (r7rs-weirdness?))
                       (loop 0)
                       (scannerError errIllegalSymbol)))
                  ((char=? c #\+)
                   (if (or (= n 1)
                           (larceny-weirdness?)
                           (r7rs-weirdness?))
                       (loop 0)
                       (scannerError errIllegalSymbol)))
                  ((char=? c #\-)
                   (if (or (= n 1)
                           (larceny-weirdness?)
                           (and (r6rs-weirdness?)
                                (char=? #\> (string-ref tokenValue 1)))
                           (and (r7rs-weirdness?)
                                ;; don't allow -1+
                                (let ((c1 (string-ref tokenValue 1)))
                                  (not (char=? c1 #\1)))))
                       (loop 0)
                       (scannerError errIllegalSymbol)))
                  ((char=? c #\.)
                   (if (or (larceny-weirdness?)
                           (r7rs-weirdness?)
                           (string=? "..." tokenValue))
                       (loop 0)
                       (scannerError errIllegalSymbol)))
                  ((char=? c #\1)
                   (if (and (larceny-weirdness?)
                            (member tokenValue '("1+" "1-")))
                       (loop 0)
                       (scannerError errIllegalSymbol)))
                  ((char=? c #\|)
                   (if (or (r7rs-weirdness?)
                           (traditional-weirdness?))
                       (symbol-elements 1 '())
                       (scannerError errIllegalSymbol)))
                  (else
                   (loop 0)))))

        ;; Parsing the interior of a symbol surrounded by vertical lines.
        ;; i is the start of the next character, and
        ;; chars is a list of the characters seen so far in reverse order.
        ;; The tokenValue was accepted by the state machine,
        ;; which saves some checking here.

        (define (symbol-elements i chars)
          (let ((c (string-ref tokenValue i)))
            (case c
             ((#\|)
              (if (not (= (+ i 1) n))
                  (scannerError errBug))
              (return (string->symbol (list->string (reverse chars)))))
             ((#\\)
              (if (not (< (+ i 1) n))
                  (scannerError errBug))
              (case (string-ref tokenValue (+ i 1))
               ((#\x)
                (call-with-values
                 (lambda () (hex-escape tokenValue (+ i 2)))
                 (lambda (sv i)
                   (symbol-elements i
                                    (cons (checked-integer->char sv)
                                          chars)))))
               ((#\a)
                (symbol-elements (+ i 2)
                                 (cons char:alarm chars)))
               ((#\b)
                (symbol-elements (+ i 2)
                                 (cons #\backspace chars)))
               ((#\t)
                (symbol-elements (+ i 2)
                                 (cons #\tab chars)))
               ((#\n)
                (symbol-elements (+ i 2)
                                 (cons #\newline chars)))
               ((#\r)
                (symbol-elements (+ i 2)
                                 (cons #\return chars)))
               ((#\| #\" #\\)
                (symbol-elements (+ i 2)
                                 (cons (string-ref tokenValue (+ i 1)) chars)))
               (else
                (scannerError errIllegalSymbol))))
             (else
              (symbol-elements (+ i 1) (cons c chars))))))

        ;; Parsing <subsequent>* with additional knowledge:
        ;; the tokenValue was accepted by the state machine.

        (define (loop i)
          (if (= i n)
              (return (string->symbol (if (port-folds-case? input-port)
                                          (string-foldcase tokenValue)
                                          tokenValue)))
              (let ((c (string-ref tokenValue i)))
                (cond ((or (char=? c #\\)
                           (char=? c #\#))
                       (slow-loop i
                                  (reverse
                                   (string->list (substring tokenValue 0 i)))
                                  (port-folds-case? input-port)))
                      ((and (char=? c #\|)
                            (not
                             (larceny-weirdness?)))
                       (scannerError errIllegalSymbol))
                      (else
                       (loop (+ i 1)))))))

        (define (slow-loop i chars fold-case?)
          (if (= i n)
              (return (string->symbol (list->string (reverse chars))))
              (let ((c (string-ref tokenValue i)))
                (cond ((char=? c #\\)
                       (cond ((and (or (larceny-weirdness?)
                                       (r6rs-weirdness?))
                                   (< (+ i 1) n)
                                   (char=? (string-ref tokenValue (+ i 1))
                                           #\x))
                              (call-with-values
                               (lambda () (hex-escape tokenValue (+ i 2)))
                               (lambda (sv i)
                                 (slow-loop i
                                            (cons (checked-integer->char sv)
                                                  chars)
                                            fold-case?))))
                             ((and (< (+ i 1) n)
                                   (larceny-weirdness?))
                              (slow-loop (+ i 2)
                                         (cons (string-ref tokenValue (+ i 1))
                                               chars)
                                         #f))
                             (else
                              (scannerError errIllegalSymbol))))
                      ((and (char=? c #\|)
                            (not (larceny-weirdness?)))
                       (scannerError errIllegalSymbol))
                      ((char=? c #\#)
                       (if (and (< (+ i 1) n)
                                (mzscheme-weirdness?)
                                (char=? (string-ref tokenValue (+ i 1))
                                        #\%))
                           (slow-loop (+ i 1) (cons c chars) fold-case?)
                           (scannerError errIllegalSymbol)))
                      (else (slow-loop (+ i 1) (cons c chars) fold-case?))))))

        (identifier-prefix)))

    (define (makeText)
      (if (r7rs-weirdness?)
          (makeStringOrText-common 'text)
          (scannerError errIllegalText)))

    ; #"..." Ascii string syntax of MzScheme

    (define (makeXString)
      (if (mzscheme-weirdness?)
          (begin (set! tokenValue
                       (substring
                        tokenValue 0 (- (string-length tokenValue) 1)))
                 (makeString))
          (parse-error '<datum> datum-starters)))
  
    ; Like append, but allows the last argument to be a non-list.
  
    (define (pseudoAppend vals terminus)
      (if (null? vals)
          terminus
          (cons (car vals)
                (pseudoAppend (cdr vals) terminus))))

    ; #.(...) read-time evaluation

    (define (sharpDot x)
      (if (traditional-weirdness?)
          (eval x)
          (parse-error '<datum> datum-starters)))

    ; Returns the source location at the start of the previous token.

    (define (sourceLocation) locationStart)

    ; #^Gsym syntax used in .fasl files

    (define (sym2global sym)
      (if (larceny-weirdness?)
          ((global-name-resolver) sym)
          (parse-error '<datum> datum-starters)))

    ; Oh, dear.  There's no excuse, but see lib/Standard/box.sch

    (define (symBox)
      (if (and (traditional-weirdness?)
               (bound? box))
          (box (parse-datum))
          (parse-error '<datum> datum-starters)))
  
    (define (symBackquote) 'quasiquote)
    (define (symQuasisyntax) 'quasisyntax)
    (define (symQuote) 'quote)
    (define (symSplicing) 'unquote-splicing)
    (define (symSyntax) 'syntax)
    (define (symUnquote) 'unquote)
    (define (symUnsyntax) 'unsyntax)
    (define (symUnsyntax-splicing) 'unsyntax-splicing)
  
    ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
    ;
    ; Action routines for SRFI 38.
    ;
    ; The shared-structures hashtable defines a mapping from
    ; indexes to fixup objects.
    ;
    ; A fixup object is a record with two mutable fields:
    ;     ready: #t if the object field is ready, else #f
    ;     value: if ready, the object that will replace the
    ;            fixup object during a post-pass
    ;
    ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

    (define (sharingDef)
      (let* ((index
              (string->number
               (substring tokenValue 1 (- (string-length tokenValue) 1))))
             (fixup (make-fixup-object index)))
        (if (not shared-structures)
            (set! shared-structures
                  (make-hashtable values =)))
        (hashtable-set! shared-structures index fixup)
        fixup))

    (define (sharingUse)
      (let* ((index
              (string->number
               (substring tokenValue 1 (- (string-length tokenValue) 1)))))
        (if (not shared-structures)
            (scannerError errSRFI38))
        (let ((fixup (hashtable-ref shared-structures index #f)))
          (if (not fixup)
              (scannerError errSRFI38))
          fixup)))

    (define (makeSharingDef fixup datum)
      (fixup-ready! fixup datum)
      datum)

    (define (makeSharingUse fixup)
      fixup)

    ;; After everything has been read, a second pass prepares
    ;; and then executes the side effects needed to recreate
    ;; the shared structure.

    (define (srfi38-postpass x)
      (let ((fixups '()))

        (define (add-fixup! fixup-object kind . rest)
          (if (fixup-ready? fixup-object)
              (set! fixups
                    (cons (cons (fixup-value fixup-object)
                                (cons kind rest))
                          fixups))
              (assertion-violation 'read-with-shared-structure
                                   "undefined index"
                                   (fixup-index fixup-object))))

        (define (postpass x)
          (cond ((pair? x)
                 (if (fixup-object? (car x))
                     (add-fixup! (car x) 'set-car! x)
                     (postpass (car x)))
                 (if (fixup-object? (cdr x))
                     (add-fixup! (cdr x) 'set-cdr! x)
                     (postpass (cdr x))))
                ((vector? x)
                 (do ((n (vector-length x))
                      (i 0 (+ i 1)))
                     ((= i n))
                   (let ((y (vector-ref x i)))
                     (if (fixup-object? y)
                         (add-fixup! y 'vector-set! x i)
                         (postpass y)))))
                (else #f)))

        (define (fixup! fixup)
          (let ((value (car fixup))
                (kind (cadr fixup))
                (container (caddr fixup))
                (rest (cdddr fixup)))
            (case kind
             ((set-car!)
              (set-car! container value))
             ((set-cdr!)
              (set-cdr! container value))
             ((vector-set!)
              (vector-set! container (car rest) value))
             (else
              (assert #f)))))

      (if shared-structures
          (begin (postpass x)
                 (for-each fixup! fixups)
                 (if (fixup-object? x)
                     (fixup-value x)
                     x))
          x)))

    ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
    ;
    ; Error procedure called by the parser.
    ;
    ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
  
    (define (parse-error nonterminal expected-terminals)
      (let* ((culprit (next-token))
             (culprit-as-string (symbol->string culprit))
             (culprit-as-string
              (if (memq culprit expected-terminals)
                  (string-append "illegal " culprit-as-string)
                  culprit-as-string))
             (msg (string-append
                   "Syntax error in line "
                   (number->string (+ 1 (port-lines-read input-port)))
                   " while parsing "
                   (symbol->string nonterminal)
                   (string #\newline)
                   "  Encountered "
                   culprit-as-string
                   " while expecting "
                   (case nonterminal
                    ((<datum> <outermost-datum> <data>)
                     "a datum")
                    (else
                     (string-append
                      (string #\newline)
                      "  "
                      (apply string-append
                             (map (lambda (terminal)
                                    (string-append " "
                                                   (symbol->string terminal)))
                                  expected-terminals)))))
                   (string #\newline))))
        (raise-r6rs-exception (make-lexical-violation)
                              'get-datum
                              msg
                              (list culprit-as-string input-port))))

    ; The list of tokens that can start a datum in R6RS or R7RS mode.

    (define datum-starters
      '(backquote
        boolean
        bvecstart
        character
        comma
        id
        lbracket
        lparen
        miscflag
        number
        quasisyntax
        quote
        sharingdef     ; R7RS only
        sharinguse     ; R7RS only
        splicing
        string
        syntax
        unsyntax
        unsyntaxsplicing
        vecstart))
  
    ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
    ;
    ; Hand-coding scanner0 makes a small but worthwhile difference.
    ;
    ; The most common characters are spaces, parentheses, newlines,
    ; semicolons, and lower case Ascii letters.
    ;
    ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
  
    ; Scanning for the start of a token.

    (define (scanner0)
      (define (loop c)
        (cond ((not (char? c))
               (accept 'eofobj))
              ((or (char=? c #\space)
                   (char=? c #\newline))
               (read-char input-port)
               (loop (peek-char input-port)))
              ((char=? c #\;)
               (scanner1))
              (else
               (if keep-source-locations?
                   (set! locationStart
                         (make-source-location input-port)))
               (state0 c))))
      (loop (peek-char input-port)))

    ; Consuming a semicolon comment.

    (define (scanner1)
      (define (loop c)
        (cond ((not (char? c))
               (accept 'eofobj))
              ((or (char=? c #\newline)
                   (char=? c #\return)
                   (char=? c (integer->char #x85))     ; Next Line, NEL
                   (char=? c (integer->char #x2028)))  ; Line Separator, LS
               (scanner0))
              (else
               (loop (read-char input-port)))))
      (loop (read-char input-port)))

    ;; The state machine is recreated for every call to get-datum,
    ;; but not for every token.

    (define state0
      (make-state-machine (lambda () (set! string_accumulator_length 0))
                          (lambda () (scanChar))
                          (lambda () (consumeChar))
                          (lambda (t) (accept t))
                          (lambda (m) (scannerError m))
                          errIncompleteToken))

    (let ((x (srfi38-postpass (parse-outermost-datum))))
      (if keep-source-locations?
          (values x locations)
          x))))

; eof
